<?php
/**
 * FileCollector - Intelligente Datei-Sammlung
 * 
 * Sammelt Dateien für das Backup mit:
 * - Automatischen Ausschlüssen (Cache, Logs, andere Backups)
 * - Größen-basierter Filterung
 * - Typ-Kategorisierung (uploads, themes, plugins, other)
 * - Heuristiken für bekannte Plugin-Verzeichnisse
 * 
 * @package JenvaBackupMigration
 * @since 2.0.0
 */

namespace JenvaBackupMigration\Backup;

if (!defined('ABSPATH')) {
    exit;
}

class FileCollector {
    
    /** @var array Standard-Ausschlüsse */
    private $default_exclusions = [
        // Cache-Verzeichnisse
        '*/cache/*',
        '*/wp-cache/*',
        '*/.cache/*',
        '*/object-cache.php',
        
        // Log-Dateien
        '*.log',
        '*/logs/*',
        '*/log/*',
        
        // Temporäre Dateien
        '*/tmp/*',
        '*/temp/*',
        '*/.tmp/*',
        '*.tmp',
        '*.temp',
        
        // Backup-Plugin-Verzeichnisse
        '*/updraftplus/*',
        '*/backwpup-*/*',
        '*/ai1wm-backups/*',
        '*/backup/*',
        '*/backups/*',
        '*/duplicator/*',
        '*/jenva-backup-migration/*',
        '*/wpvivid-backuprestore/*',
        '*/developer/*',
        
        // Sicherheits-Plugin-Verzeichnisse
        '*/wordfence/*',
        '*/sucuri/*',
        '*/really-simple-ssl/*',
        
        // Entwicklungs-Dateien
        '*/node_modules/*',
        '*/.git/*',
        '*/.svn/*',
        '*/vendor/*',
        
        // WordPress Upgrade-Dateien
        '*/upgrade/*',
        
        // Große Media-Dateien (optional)
        // '*.mp4',
        // '*.mov',
        // '*.avi',
    ];
    
    /** @var array Gesammelte Dateien */
    private $files = [];
    
    /** @var int Ausgeschlossene Dateien */
    private $excluded_count = 0;
    
    /** @var int Ausgeschlossene Größe */
    private $excluded_size = 0;
    
    /** @var array Aktive Ausschlüsse */
    private $exclusions = [];
    
    /**
     * Sammelt alle Dateien für das Backup
     * 
     * @param array $additional_exclusions Zusätzliche Ausschlüsse
     * @return array Datei-Liste
     */
    public function collect(array $additional_exclusions = []): array {
        $this->files = [];
        $this->excluded_count = 0;
        $this->excluded_size = 0;
        $this->exclusions = array_merge($this->default_exclusions, $additional_exclusions);
        
        // wp-content Verzeichnisse sammeln
        $this->collectUploads();
        $this->collectThemes();
        $this->collectMuPlugins();
        
        // Andere wichtige Dateien
        $this->collectRootFiles();
        
        return $this->files;
    }
    
    /**
     * Sammelt Dateien aus einem Verzeichnis
     * 
     * @param string $directory Verzeichnis
     * @return array Dateien
     */
    public function collectDirectory(string $directory): array {
        $files = [];
        
        if (!is_dir($directory)) {
            return $files;
        }
        
        $iterator = new \RecursiveIteratorIterator(
            new \RecursiveDirectoryIterator($directory, \RecursiveDirectoryIterator::SKIP_DOTS),
            \RecursiveIteratorIterator::SELF_FIRST
        );
        
        foreach ($iterator as $file) {
            if ($file->isDir()) {
                continue;
            }
            
            $path = $file->getPathname();
            $relative = str_replace($directory . '/', '', $path);
            
            $files[] = [
                'path' => $path,
                'relative' => $relative,
                'size' => $file->getSize(),
                'type' => $this->determineType($path),
            ];
        }
        
        return $files;
    }
    
    /**
     * Sammelt Uploads
     */
    private function collectUploads(): void {
        $upload_dir = wp_upload_dir();
        $base_dir = $upload_dir['basedir'];
        
        if (!is_dir($base_dir)) {
            return;
        }
        
        $this->collectFromDirectory($base_dir, 'uploads', 'uploads');
    }
    
    /**
     * Sammelt Themes
     */
    private function collectThemes(): void {
        $themes_dir = get_theme_root();
        
        if (!is_dir($themes_dir)) {
            return;
        }
        
        // Nur aktives Theme und Parent-Theme
        $active_theme = get_stylesheet();
        $parent_theme = get_template();
        
        $active_dir = $themes_dir . '/' . $active_theme;
        if (is_dir($active_dir)) {
            $this->collectFromDirectory($active_dir, 'themes', 'themes/' . $active_theme);
        }
        
        // Parent-Theme wenn unterschiedlich
        if ($parent_theme !== $active_theme) {
            $parent_dir = $themes_dir . '/' . $parent_theme;
            if (is_dir($parent_dir)) {
                $this->collectFromDirectory($parent_dir, 'themes', 'themes/' . $parent_theme);
            }
        }
    }
    
    /**
     * Sammelt MU-Plugins
     */
    private function collectMuPlugins(): void {
        $mu_dir = WPMU_PLUGIN_DIR;
        
        if (!is_dir($mu_dir)) {
            return;
        }
        
        $this->collectFromDirectory($mu_dir, 'mu-plugins', 'mu-plugins');
    }
    
    /**
     * Sammelt wichtige Root-Dateien
     */
    private function collectRootFiles(): void {
        $root_files = [
            '.htaccess',
            'wp-config.php',
            'robots.txt',
            'favicon.ico',
        ];
        
        foreach ($root_files as $file) {
            $path = ABSPATH . $file;
            
            if (file_exists($path) && is_file($path)) {
                $this->files[] = [
                    'path' => $path,
                    'relative' => $file,
                    'size' => filesize($path),
                    'type' => 'root',
                ];
            }
        }
    }
    
    /**
     * Sammelt Dateien aus einem Verzeichnis
     */
    private function collectFromDirectory(string $directory, string $type, string $prefix): void {
        if (!is_dir($directory)) {
            return;
        }
        
        $iterator = new \RecursiveIteratorIterator(
            new \RecursiveDirectoryIterator($directory, \RecursiveDirectoryIterator::SKIP_DOTS),
            \RecursiveIteratorIterator::SELF_FIRST
        );
        
        foreach ($iterator as $file) {
            if ($file->isDir()) {
                continue;
            }
            
            $path = $file->getPathname();
            
            // Ausschlüsse prüfen
            if ($this->isExcluded($path)) {
                $this->excluded_count++;
                $this->excluded_size += $file->getSize();
                continue;
            }
            
            $relative = $prefix . '/' . str_replace($directory . '/', '', $path);
            
            $this->files[] = [
                'path' => $path,
                'relative' => $relative,
                'size' => $file->getSize(),
                'type' => $type,
            ];
        }
    }
    
    /**
     * Prüft ob eine Datei ausgeschlossen werden soll
     */
    private function isExcluded(string $path): bool {
        foreach ($this->exclusions as $pattern) {
            if ($this->matchPattern($path, $pattern)) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Pattern-Matching (vereinfachtes Glob)
     */
    private function matchPattern(string $path, string $pattern): bool {
        // Konvertiere Glob-Pattern zu Regex
        $regex = str_replace(
            ['*', '?', '/'],
            ['.*', '.', '\/'],
            $pattern
        );
        
        return preg_match('/^' . $regex . '$/i', $path) === 1 ||
               preg_match('/' . $regex . '$/i', $path) === 1;
    }
    
    /**
     * Bestimmt den Typ einer Datei
     */
    private function determineType(string $path): string {
        if (strpos($path, '/uploads/') !== false) {
            return 'uploads';
        }
        if (strpos($path, '/themes/') !== false) {
            return 'themes';
        }
        if (strpos($path, '/plugins/') !== false) {
            return 'plugins';
        }
        if (strpos($path, '/mu-plugins/') !== false) {
            return 'mu-plugins';
        }
        
        return 'other';
    }
    
    /**
     * Fügt einen Ausschluss hinzu
     */
    public function addExclusion(string $pattern): void {
        $this->exclusions[] = $pattern;
    }
    
    /**
     * Getter für ausgeschlossene Anzahl
     */
    public function getExcludedCount(): int {
        return $this->excluded_count;
    }
    
    /**
     * Getter für ausgeschlossene Größe
     */
    public function getExcludedSize(): int {
        return $this->excluded_size;
    }
    
    /**
     * Gibt Statistiken zurück
     */
    public function getStats(): array {
        $stats = [
            'total_files' => count($this->files),
            'total_size' => 0,
            'by_type' => [],
            'excluded_files' => $this->excluded_count,
            'excluded_size' => $this->excluded_size,
        ];
        
        foreach ($this->files as $file) {
            $stats['total_size'] += $file['size'];
            
            $type = $file['type'];
            if (!isset($stats['by_type'][$type])) {
                $stats['by_type'][$type] = ['count' => 0, 'size' => 0];
            }
            $stats['by_type'][$type]['count']++;
            $stats['by_type'][$type]['size'] += $file['size'];
        }
        
        return $stats;
    }
    
    /**
     * Formatiert Bytes als lesbare Größe
     */
    public static function formatSize(int $bytes): string {
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];
        $i = 0;
        
        while ($bytes >= 1024 && $i < count($units) - 1) {
            $bytes /= 1024;
            $i++;
        }
        
        return round($bytes, 2) . ' ' . $units[$i];
    }
}

